//Temporal Anti Aliasing from Pissang
//https://github.com/pissang/claygl-advanced-renderer/blob/master/src/TAA.glsl

struct temporalSettings{
  float weight;
  float sharpening;
  float softness;

  float velocityFull;
  float velocityNone;
  float velocityMax;
  float velocitySpan;
};

temporalSettings renderTemporalSettings(float depth){
  temporalSettings t;

  //Temporal settings
  t.weight = depth < 0.7 ? 0 : 0.97 * TEMPORAL_WEIGHT;
  t.sharpening = TEMPORAL_SHARPERNING;
  t.softness = TEMPORAL_SOFTNESS;

  //Motion blur settings
  t.velocityFull = 2;
  t.velocityNone = 15;
  t.velocityMax = 0.025;
  t.velocitySpan = t.velocityNone - t.velocityFull;

  return t;
}

vec3 renderCurrentColor(vec2 coord){
  return toGamma(texture2D(colortex0, coord + jitter * 0.5).rgb);
}

vec3 renderPreviousColor(vec2 coord, temporalSettings t){
  return toGamma(texture2D(colortex4, coord).rgb);
}

vec3 renderClosestFragment(sampler2D depth, vec2 uv, vec2 pixelSize) {
	vec2 dd = abs(pixelSize);
	vec2 du = vec2(dd.x, 0.0);
	vec2 dv = vec2(0.0, dd.y);

	vec3 dtl = vec3(-1, -1, texture2D(depth, uv - dv - du).x);
	vec3 dtc = vec3( 0, -1, texture2D(depth, uv - dv).x);
	vec3 dtr = vec3( 1, -1, texture2D(depth, uv - dv + du).x);

	vec3 dml = vec3(-1, 0, texture2D(depth, uv - du).x);
	vec3 dmc = vec3( 0, 0, texture2D(depth, uv).x);
	vec3 dmr = vec3( 1, 0, texture2D(depth, uv + du).x);

	vec3 dbl = vec3(-1, 1, texture2D(depth, uv + dv - du).x);
	vec3 dbc = vec3( 0, 1, texture2D(depth, uv + dv).x);
	vec3 dbr = vec3( 1, 1, texture2D(depth, uv + dv + du).x);

	vec3 dmin = dtl;
       dmin = dmin.z > dtc.z ? dtc : dmin;
       dmin = dmin.z > dtr.z ? dtr : dmin;

       dmin = dmin.z > dml.z ? dml : dmin;
       dmin = dmin.z > dmc.z ? dmc : dmin;
       dmin = dmin.z > dmr.z ? dmr : dmin;

       dmin = dmin.z > dbl.z ? dbc : dmin;
       dmin = dmin.z > dbc.z ? dbc : dmin;
       dmin = dmin.z > dbc.z ? dbc : dmin;

	return vec3(uv + dd.xy * dmin.xy, dmin.z);
}

vec3 renderClipAABB(vec3 q, vec3 aabbMin, vec3 aabbMax) {
	vec3 p_clip = 0.5 * (aabbMax + aabbMin);
	vec3 e_clip = 0.5 * (aabbMax - aabbMin);

	vec3 v_clip = q - p_clip;
	vec3 v_unit = v_clip.xyz / e_clip;
	vec3 a_unit = abs(v_unit);

	float ma_unit = max(a_unit.x, max(a_unit.y, a_unit.z));

    return ma_unit > 1 ? p_clip + v_clip / ma_unit : q;
}

vec3 renderMotionBlur(temporalSettings t, vec2 coord, vec2 velocity, float depth){
  const int steps = 2;

  vec2 sampleOffset = velocity * 0.1 / frameTime * 0.055 * MOTION_BLUR_STRENGTH;
       sampleOffset = length(sampleOffset) > t.velocityMax ? clamp(sampleOffset, vec2(-t.velocityMax), vec2(t.velocityMax)) : sampleOffset; 
       sampleOffset = depth < 0.7 ? vec2(0) : sampleOffset;

  float dither = bayer64(gl_FragCoord.xy);

  vec3 color = vec3(0);
  float weight = 0;

  for(int i = -steps; i <= steps; ++i){
    float sampleWeight = 1 - abs(i) / steps;
    vec2 sampleCoord = coord + sampleOffset * ((i + dither) / steps);

    color += renderCurrentColor(sampleCoord);
    weight += 1;
  }
  
  #ifdef MOTION_BLUR
  return color / weight;
  #else
  return renderCurrentColor(coord);
  #endif
}

vec3 renderTemporalProjection(vec2 coord){
  vec3 closest = renderClosestFragment(depthtex0, coord, rPixel);
  float depth = closest.z;
  vec2 velocity = renderVelocity(closest);

  //if(velocity.x < 0 || velocity.y < 0 || velocity.x > 1 || velocity.y > 1) return toLinear(renderCurrentColor(coord + jitter * 0.5));

  temporalSettings t = renderTemporalSettings(depth);

  float trust = 1 - clamp(length(velocity * pixel) - t.velocityFull, 0, t.velocitySpan) / t.velocitySpan;

  vec3 motion = renderMotionBlur(t, coord, velocity, depth);
  vec3 current = renderCurrentColor(coord);
  vec3 previous = renderPreviousColor(coord - velocity, t);

	vec2 dx = vec2(rPixel.x, 0.0);
	vec2 dy = vec2(0.0, rPixel.y);

	vec3 ctl = renderCurrentColor(coord - dy - dx);
	vec3 ctc = renderCurrentColor(coord - dy);
	vec3 ctr = renderCurrentColor(coord - dy + dx);

	vec3 cml = renderCurrentColor(coord - dx);
	vec3 cmc = renderCurrentColor(coord);
	vec3 cmr = renderCurrentColor(coord + dx);

	vec3 cbl = renderCurrentColor(coord + dy - dx);
	vec3 cbc = renderCurrentColor(coord + dy);
	vec3 cbr = renderCurrentColor(coord + dy + dx);

	vec3 cmin = min(ctl, min(ctc, min(ctr, min(cml, min(cmc, min(cmr, min(cbl, min(cbc, cbr))))))));
	vec3 cmax = max(ctl, max(ctc, max(ctr, max(cml, max(cmc, max(cmr, max(cbl, max(cbc, cbr))))))));

	vec3 cmin5 = min(ctc, min(cml, min(cmc, min(cmr, cbc))));
	vec3 cmax5 = max(ctc, max(cml, max(cmc, max(cmr, cbc))));

	vec3 cavg = (ctl + ctc + ctr + cml + cmc + cmr + cbl + cbc + cbr) / 9;
	vec3 cavg5 = (ctc + cml + cmc + cmr + cbc) / 5;

	cmin = 0.5 * (cmin + cmin5);
	cmax = 0.5 * (cmax + cmax5);
	cavg = 0.5 * (cavg + cavg5);

  previous = renderClipAABB(previous, cmin, cmax);
  
  vec2 velCoord = coord - velocity;
  vec2 texel = abs(fract(velocity * pixel) - 0.5);

  float weight = clamp01(velCoord) != velCoord ? 0.0 : t.weight * sqrt(texel.x * texel.y) * 0.25 + (1 - 0.25);

  current += (1 - exp2(-(current - clamp(cavg, cmin, cmax)))) * 1.5 * t.sharpening;
  current = mix(current, mix(cmin, cmax, cavg), vec3(t.softness));

  #ifdef TEMPORAL_AA
    #ifdef MOTION_BLUR
      vec3 color = mix(motion, mix(current, previous, weight), trust);
    #else
      vec3 color = mix(current, previous, weight);
    #endif
  #else
    #ifdef MOTION_BLUR
      vec3 color = mix(motion, renderCurrentColor(coord), trust);
    #else
      vec3 color = renderCurrentColor(coord);
    #endif
  #endif

  return toLinear(color);
}