#if defined vertex
out vec2 coord;
out vec2 lmcoord;
out vec2 midcoord;

out float parallaxDepth;
out vec2 parallaxResolution;
out vec2 parallaxSize;

out vec3 position;
out vec3 worldPosition;
out vec3 normal;
out vec3 tangentVector;
out vec3 tangentView;

out vec4 color;

flat out float mask;
flat out float material;

out mat3 tbn;

#include "/program/gbuffers/common/curve.vsh"
#include "/program/gbuffers/common/displacement.vsh"
#include "/program/gbuffers/common/material.vsh"
#include "/program/gbuffers/common/tbn.vsh"

void renderParallaxData(vec2 coord, inout float parallaxDepth, inout vec2 parallaxResolution, inout vec2 parallaxSize){
  vec2 resolution = vec2(textureSize2D(texture, 0));

  parallaxSize = abs(coord - mc_midTexCoord.xy) * 2.0;
  parallaxDepth = length(parallaxSize);
}

void main(){
  coord = gl_MultiTexCoord0.xy;
  midcoord = mc_midTexCoord.xy;
	
  #if MC_VERSION >= 15500
  lmcoord = gl_MultiTexCoord2.xy / 255.0;
  #else
  lmcoord = gl_MultiTexCoord1.xy / 255.0;
  #endif
  
  renderParallaxData(coord, parallaxDepth, parallaxResolution, parallaxSize);

  position = transMAD(gl_ModelViewMatrix, gl_Vertex.xyz);
  worldPosition = renderVertexDisplacement(transMAD(gbufferModelViewInverse, position));
  worldPosition = renderEarthCurvature(worldPosition);
  position = transMAD(gbufferModelView, worldPosition);
  normal = gl_NormalMatrix * gl_Normal;

  color = gl_Color;

  renderMaterial(material, mask, lmcoord.x);

  tbn = renderTBN(gl_Normal);

  vec2 resolution = vec2(textureSize2D(texture, 0));
  mat3 parallaxTBN = mat3(tbn[0] * resolution.y / resolution.x, tbn[1], tbn[2]);

  tangentVector = normalize((worldPosition + gbufferModelViewInverse[3].xyz) * parallaxTBN);
  tangentView = (position * gl_NormalMatrix) * parallaxTBN;

  #if defined hand
    #ifdef MOVING_HAND
      position.y += sin(frameTimeCounter * MOVING_HAND_SPEED) * 0.03;
      position.z -= cos(frameTimeCounter * MOVING_HAND_SPEED) * 0.003;
    #endif

  gl_Position = position.xyzz * diagonal4(gl_ProjectionMatrix) + gl_ProjectionMatrix[3];
  #else
  gl_Position = position.xyzz * diagonal4(gbufferProjection) + gbufferProjection[3];
  #endif
  gl_Position.xy = jitter * gl_Position.w +  gl_Position.xy;
}
#elif defined fragment
in vec2 coord;
in vec2 lmcoord;
in vec2 midcoord;

in float parallaxDepth;
in vec2 parallaxResolution;
in vec2 parallaxSize;

in vec3 position;
in vec3 worldPosition;
in vec3 normal;
in vec3 tangentVector;
in vec3 tangentView;

in vec4 color;

flat in float mask;
flat in float material;

in mat3 tbn;

bool isWater = material == 1006 || material == 5001;
bool isIce = material == 1007;
bool isStainedGlass = material == 1008;
bool isSlime = material == 10081;

mat2 texD = mat2(dFdx(coord), dFdy(coord));

#if defined water
#include "/program/gbuffers/common/waterNormal.glsl"
#endif

#if defined terrain
#include "/program/gbuffers/common/puddles.glsl"
#endif

#include "/program/gbuffers/common/pom.glsl"

vec4 renderAlbedo(vec4 data0){
  #if !defined water
  if(data0.a < 0.301) discard;
  #endif

  #if defined basic
  vec4 albedo = color;
       albedo.a = albedo.a * 0.5 + 0.5;
  #else
  vec4 albedo = data0 * color;
  #endif

  #if defined water
       albedo = isWater ? vec4(1.0) : albedo;
  #elif defined textured
       albedo.a = 0;
  #endif

  return albedo;
}

float renderNormal(vec3 data1){
  #if defined terrain
  vec3 normal = data1;

       normal.xy = normal.xy * 2 - 1;
       normal.z = sqrt(1 - dot(normal.xy, normal.xy));
  #else
  vec3 normal = data1 * 2.0 - 1.0;
  #endif

  vec3 flatNormal = clamp(cross(dFdx(worldPosition), dFdy(worldPosition)), -1.0, 1.0);
  //if(dot(flatNormal, normal) < 0.0) normal += flatNormal;

  #if defined water
  vec3 wave = worldPosition + cameraPosition;
  #if WATER_PARALLAX == 1 || WATER_PARALLAX == 2
       wave.xz = renderParallaxWater(wave, -tangentVector);
  #endif

  normal = isWater ? renderWaveNormal(wave) : normal;

  #if WATER_PARALLAX == 2
  if(isWater){
    vec3 viewSpace = transMAD(gbufferModelView, wave - cameraPosition);
  	vec3 clipSpace = (diagonal3(gbufferProjection) * viewSpace + gbufferProjection[3].xyz) / -viewSpace.z * 0.5 + 0.5;
  	gl_FragDepth = isEyeInWater == 0 ? clipSpace.z : gl_FragCoord.z;
  } else {
    gl_FragDepth = gl_FragCoord.z;
  }
  #endif
  #endif

  return EncodeNormal(tbn * normal);
}

float renderLightmap(){
  float dither = dither64(gl_FragCoord.xy) / 255;

  float torchlight = clamp01(lmcoord.x + dither);
  float lightmap = clamp01(lmcoord.y + dither);
        
  return EncodeVec2(torchlight, lightmap);
}

float rendereOldf0(float metalness) {
	float f0 = 1.0;

	switch(int(material)) {
		case 1016: f0 = 0.97; break; // Gold
		case 1015: f0 = 0.46; break; // Iron

		default: break;
	}

	return f0 * metalness;
}

float renderSpecular(vec4 data2, inout vec3 albedo){
  vec4 spec = data2;

  #if SPEC_FORMAT == 0
  float roughness = 1.0 - spec.x;
  float f0 = spec.y;
  float sss = 1 - spec.a;
  #else
  float roughness = 1.0 - spec.x;
  float f0 = rendereOldf0(spec.y);
  float sss = spec.z;
  #endif

  float emissive = float(material == 1009 ||
                   material == 1010 ||
                   material == 1011 ||
                   material == 1012 ||
                   material == 1013 ||
                   material == 1014);

  //albedo = albedo + emissive * sss;

  #if defined terrain
    #ifdef PUDDLES
      roughness = mix(roughness, 0.03, renderPuddlesRain(worldPosition + cameraPosition));
      f0 = mix(f0, 0.021, renderPuddlesRain(worldPosition + cameraPosition));
    #endif
  #endif

  #if defined water
    if(isWater){
      roughness = 0.021;
      f0 = 0.26;
    }

    if(isIce){
      roughness = 0.018;
      f0 = 0.22;
    }

    #ifdef STAINED_GLASS_REFLECTION
      roughness = isStainedGlass || isSlime? 0.042 : roughness;
      f0 = isStainedGlass || isSlime ? 0.52 : f0;
    #else
      roughness = isSlime ? 0.042 : roughness;
      f0 = isSlime ? 0.52 : f0;
    #endif
  #endif

  #if defined terrain || defined hand
    #if SPEC_FORMAT == 0
      int matID = int(f0 * 255);

      if(matID == 230 || matID == 231 || matID == 232 || matID == 233 || matID == 234 || matID == 236 || matID == 237){
        roughness = 0.06; 
        f0 = 0.75;
      } else if(matID == 235){
        roughness = 0.113;
        f0 = 1;
      }
    
    #endif
  #endif

  return EncodeVec2(roughness, f0);
}

void main(){
  #if defined water
  	if(!gl_FrontFacing) discard;
  #endif

  uniformData d = renderUniformData(coord);

  vec4 albedo = renderAlbedo(d.data0);
  
  float normal = renderNormal(d.data1.xyz);
  float lightmap = renderLightmap();
  float specular = renderSpecular(d.data2, albedo.rgb);

  /* DRAWBUFFERS:02 */
  gl_FragData[0] = albedo;
  
  #if defined entities
  gl_FragData[1] = vec4(normal, lightmap, specular, 1);
  #elif defined basic
  gl_FragData[1] = vec4(0.0, EncodeVec2(0.0, 0.0), EncodeVec2(0.0, 0.0), EncodeVec2(1.0, 1.0));
  #else
  gl_FragData[1] = vec4(normal, lightmap, specular, EncodeVec2(1.0 - mask, clamp01(d.shadow + 0.4)));
  #endif
}
#endif
